<?php

$needsObj = new Needs();
class Needs
{
    private $mDb;
    private $mConfig;
    private $mlang;

    public function __construct($db, $config, $lang)
    {
        $this->mDb = $db;
        $this->mConfig = $config;
        $this->mlang = $lang;
    }

    // Get search data from database securely
    private function getSearchableItems()
    {
        try {
            $query = "SELECT name FROM product_langs";
            $stmt = $this->mDb->connect()->prepare($query);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_COLUMN);
        } catch (PDOException $e) {
            error_log("Database error: " . $e->getMessage()); // Log error
            return []; // Avoid exposing errors
        }
    }

    public function substringSearch($search)
    {
        $search = $this->sanitizeInput($search);
        if (!$search) return []; // Return empty array if input is invalid

        $items = $this->getSearchableItems();
        return array_filter($items, function ($sentence) use ($search) {
            $normalizedSentence = $this->normalizeText($sentence);
            $words = explode(" ", $normalizedSentence);

            foreach ($words as $word) {
                $distance = levenshtein($search, $word);
                if ($distance <= 2 || mb_stripos($word, $search) !== false) {
                    return true; 
                }
            }
            return false;
        });
    }

    // Soundex Matching - Securely checks similar-sounding words
    public function soundexSearch($search)
    {
        $search = $this->sanitizeInput($search);
        if (!$search) return [];

        $items = $this->getSearchableItems();
        $searchSoundex = soundex($search);
        return array_filter($items, fn ($item) => soundex($item) === $searchSoundex);
    }

    // Combined Fuzzy Search - Securely finds close matches
    public function fuzzySearch($search, $levThreshold = 2)
    {
        $search = $this->sanitizeInput($search);
        if (!$search) return [];

        $items = $this->getSearchableItems();
        $results = [];

        foreach ($items as $item) {
            $levDist = levenshtein(strtolower($search), strtolower($item));
            $soundexMatch = (soundex($search) == soundex($item));
            $substringMatch = (stripos($item, $search) !== false);

            if ($levDist <= $levThreshold || $soundexMatch || $substringMatch) {
                $results[$item] = $levDist;
            }
        }

        asort($results);
        return array_keys($results);
    }

    // Normalize text for secure processing
    private function normalizeText($text)
    {
        $text = mb_strtolower($text, 'UTF-8');
        return trim(preg_replace('/[^\p{Arabic}a-zA-Z0-9 ]/u', '', $text));
    }

    // Sanitize input to prevent XSS and SQL Injection
    private function sanitizeInput($input)
    {
        $input = trim($input);
        return filter_var($input, FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_HIGH);
    }
}

}